const puppeteer = require('puppeteer');
const fs = require('fs');
const ejs = require('ejs');

// Get command-line arguments passed from Java
const jsonDataPath = process.argv[2];
const templatePath = process.argv[3];
const outputPdfPath = process.argv[4];
const configPath = process.argv[5];

try {
    // Read data from JSON file
    const data = JSON.parse(fs.readFileSync(jsonDataPath, 'utf8'));

    const imagePath = data['ImagePath'];

    // Convert image file paths to base64 data URLs
    if (imagePath) {
        for (const key in imagePath) {
            if (key !== 'id' && imagePath[key]) {
                const imageSrc = imagePath[key];
                
                // Check if it's a URL (http:// or https://)
                const isUrl = imageSrc.startsWith('http://') || imageSrc.startsWith('https://');
                
                // Only convert local file paths to base64, skip URLs
                if (!isUrl) {
                    try {
                        // Read the image file
                        const imageBuffer = fs.readFileSync(imageSrc);
                        
                        // Determine MIME type based on file extension
                        const ext = imageSrc.split('.').pop().toLowerCase();
                        const mimeTypes = {
                            'jpg': 'image/jpeg',
                            'jpeg': 'image/jpeg',
                            'png': 'image/png',
                            'gif': 'image/gif',
                            'svg': 'image/svg+xml',
                            'webp': 'image/webp'
                        };
                        const mimeType = mimeTypes[ext] || 'image/jpeg';
                        
                        // Convert to base64 data URL
                        const base64Image = `data:${mimeType};base64,${imageBuffer.toString('base64')}`;
                        
                        // Replace the file path with base64 data URL
                        imagePath[key] = base64Image;
                    } catch (imgError) {
                        console.error(`Error reading image file ${imageSrc}:`, imgError.message);
                        // Keep the original path if there's an error
                    }
                }
                // If it's a URL, leave it unchanged
            }
        }
    }

    // Read config file for Puppeteer options
    const config = JSON.parse(fs.readFileSync(configPath, 'utf8'));

    // Render EJS template with data
    ejs.renderFile(templatePath, data, {}, async (err, html) => {
        if (err) {
            console.error('Error rendering template:', err.message);
            process.exit(1); // Exit with an error code
        }

        // Launch Puppeteer and generate PDF
        try {
            const browser = await puppeteer.launch({
                headless: config.headless ?? true, // Default to true if not provided
                args: ['--no-sandbox', '--disable-setuid-sandbox'],
            });

            const page = await browser.newPage();
            await page.setContent(html, { waitUntil: 'networkidle0' });

            // Extract PDF options from config
            const pdfOptions = {
                path: outputPdfPath, // File path where the PDF will be saved
                format: config.paperSize ?? 'A4', // Paper size, e.g., 'A4', 'Letter', etc.
                landscape: config.landscape ?? false, // Orientation: portrait (default) or landscape
                printBackground: config.printBackground ?? true, // Print CSS backgrounds (default: true)
                margin: config.margin ?? {}, // Margins for the PDF (in mm, cm, or px)

                scale: config.scale ?? 1, // Scales the rendering of the page (default: 1)
                displayHeaderFooter: config.displayHeaderFooter ?? false, // Display header and footer (default: false)
                headerTemplate: config.headerTemplate ?? '', // HTML template for the header
                footerTemplate: config.footerTemplate ?? '', // HTML template for the footer
                preferCSSPageSize: config.preferCSSPageSize ?? false, // Use @page CSS size over 'format' (default: false)
                timeout: config.timeout ?? 60000, // Maximum time in ms for rendering (default: 60000)
                width: config.width ?? undefined, // Custom width for the PDF (overrides 'format')
                height: config.height ?? undefined, // Custom height for the PDF (overrides 'format')
                omitBackground: config.omitBackground ?? false, // Hide background graphics (default: false)
                pageRanges: config.pageRanges ?? '', // Page ranges to include, e.g., '1-5' (default: all pages)
            };

            await page.pdf(pdfOptions);

            await browser.close();
            console.log('PDF generated at:', outputPdfPath);
            process.exit(0); // Exit with success code
        } catch (error) {
            console.error('Error generating PDF:', error.message);
            process.exit(1); // Exit with an error code
        }
    });
} catch (error) {
    console.error('Error initializing script:', error.message);
    process.exit(1); // Exit with an error code
}