const puppeteer = require('puppeteer');
const fs = require('fs');
const ejs = require('ejs');

// Get command-line arguments passed from Java
const jsonDataPath = process.argv[2];
const templatePath = process.argv[3];
const outputPdfPath = process.argv[4];
const configPath = process.argv[5];

console.log('Starting PDF generation process...');
console.log(`JSON Data: ${jsonDataPath}`);
console.log(`Template: ${templatePath}`);
console.log(`Output PDF: ${outputPdfPath}`);
console.log(`Config: ${configPath}`);

// Immediately flush console output
process.stdout.write('Reading data and config files...\n');

try {
    // Read data from JSON file
    const data = JSON.parse(fs.readFileSync(jsonDataPath, 'utf8'));
    console.log('Successfully loaded JSON data');

    // Read config file for Puppeteer options
    const config = JSON.parse(fs.readFileSync(configPath, 'utf8'));
    console.log('Successfully loaded config');

    // Render EJS template with data
    console.log('Rendering EJS template...');
    ejs.renderFile(templatePath, data, {}, async (err, html) => {
        if (err) {
            console.error('Error rendering template:', err);
            process.exit(1);
            return;
        }

        console.log('EJS template rendered successfully');

        // Launch Puppeteer and generate PDF
        try {
            console.log('Launching Puppeteer browser...');
            const browser = await puppeteer.launch({
                headless: config.headless !== undefined ? config.headless : true,
                args: ['--no-sandbox', '--disable-setuid-sandbox'],
            });
            console.log('Browser launched');

            console.log('Creating new page...');
            const page = await browser.newPage();
            console.log('Setting page content...');
            await page.setContent(html, { waitUntil: 'networkidle0' });
            console.log('Page content set');

            const pdfOptions = {
                path: outputPdfPath, // File path where the PDF will be saved
                format: config.paperSize ?? 'A4', // Paper size, e.g., 'A4', 'Letter', etc.
                landscape: config.landscape ?? false, // Orientation: portrait (default) or landscape
                printBackground: config.printBackground ?? true, // Print CSS backgrounds (default: true)
                margin: config.margin ?? {}, // Margins for the PDF (in mm, cm, or px)
                scale: config.scale ?? 1, // Scales the rendering of the page (default: 1)
                displayHeaderFooter: config.displayHeaderFooter ?? false, // Display header and footer (default: false)
                headerTemplate: config.headerTemplate ?? '', // HTML template for the header
                footerTemplate: config.footerTemplate ?? '', // HTML template for the footer
                preferCSSPageSize: config.preferCSSPageSize ?? false, // Use @page CSS size over 'format' (default: false)
                timeout: config.timeout ?? 30000, // Maximum time in ms for rendering (default: 30000)
                width: config.width ?? undefined, // Custom width for the PDF (overrides 'format')
                height: config.height ?? undefined, // Custom height for the PDF (overrides 'format')
                omitBackground: config.omitBackground ?? false, // Hide background graphics (default: false)
                pageRanges: config.pageRanges ?? '', // Page ranges to include, e.g., '1-5' (default: all pages)
            };

            console.log('Generating PDF with options:', JSON.stringify(pdfOptions, null, 2));
            await page.pdf(pdfOptions);
            console.log('PDF generation complete');

            await browser.close();
            console.log('Browser closed');
            console.log('PDF generated successfully at:', outputPdfPath);
            process.exit(0);
        } catch (error) {
            console.error('Error generating PDF:', error);
            process.exit(1);
        }
    });
} catch (error) {
    console.error('Error in initial setup:', error);
    process.exit(1);
}